(*
===============================================================================================

LinkList

===============================================================================================
Completely encapsulated Doubly Linked List
using Delphi CLASSes and Polymorphism.

- The nodes of your list MUST BE inherited of TListNode
- In order to use "Ensort" you have to override TLinkList.Compare

* Please send me a postcard of your city when you use this unit
* Please send comments and bug reports/fixes to my E-Mail address sh@heymann-net.de

===============================================================================================
You can use this unit for everything you want to. But use it at your own risk. 
I am not responsible for any damage caused by use of or reliance on this code.

No copyright, no charge, no warranty, no support.
===============================================================================================
Author:  Stefan Heymann
         Eschenweg 3
         72076 Tbingen
         Germany
         
URL:     www.destructor.de
E-Mail:  stefan@destructor.de         

You can use this code according to the Destructor Source Licence (DSL)
===============================================================================================
1999-06-26  HeySt  Fixed a bug in the InsertList method
2003-11-17  HeySt  Destructor-V2 release
2014-04-14  HeySt  Made the Compare method non-abstract to get rid of compiler warning
*)

UNIT LinkList;

INTERFACE

TYPE
  TListNode = CLASS
              PROTECTED
                FNext : TListNode;   (* Pointer to next     node of list *)
                FPrev : TListNode;   (* Pointer to previous node of list *)
              PUBLIC
                CONSTRUCTOR Create;
                PROPERTY Next : TListNode READ FNext;
                PROPERTY Prev : TListNode READ FPrev;
              END;

  TLinkList = CLASS (TListNode)
              PRIVATE
                FStartNode : TListNode;   (* Pointer to first node of list *)
                FEndNode   : TListNode;   (* Pointer to last  node of list *)
                FCount     : LONGINT;     (* Number of nodes in the list   *)
              PUBLIC
                CONSTRUCTOR Create;
                DESTRUCTOR  Destroy;          OVERRIDE;

                FUNCTION  IsEmpty    : BOOLEAN;
                FUNCTION  CountNodes : LONGINT;

                PROPERTY  First : TListNode READ FStartNode;
                PROPERTY  Last  : TListNode READ FEndNode;
                PROPERTY  Count : LONGINT   READ FCount;
                FUNCTION  At      (Index0 : LONGINT)   : TListNode;
                FUNCTION  IndexOf (Node   : TListNode) : LONGINT;

                PROCEDURE Append      (Node : TListNode);
                PROCEDURE Insert      (InsertAfterNode, Node : TListNode);
                PROCEDURE InsertFirst (Node : TListNode);
                PROCEDURE InsertList  (InsertAfterNode : TListNode; TheList : TLinkList);

                PROCEDURE Extract     (Node : TListNode);
                PROCEDURE Delete      (Node : TListNode);
                PROCEDURE Clear;

                PROCEDURE Ensort      (Node : TListNode);
                FUNCTION  Compare     (Node1, Node2: TListNode) : INTEGER;  VIRTUAL;

                PROCEDURE StartScan (VAR ScanNode);
                FUNCTION  Scan      (VAR ScanNode) : BOOLEAN;
              END;

(*
===============================================================================================
IMPLEMENTATION
===============================================================================================
*)

IMPLEMENTATION


(*
===============================================================================================
TListNode
===============================================================================================
*)

CONSTRUCTOR TListNode.Create;
            (* Initializes the pointers to NIL *)
BEGIN
  INHERITED Create;
  FNext := NIL;
  FPrev := NIL;
  (* I know that TObject.Create does this. But this is still Pascal - isn't it? *)
END;


(*
===============================================================================================
TLinkList
===============================================================================================
*)

CONSTRUCTOR TLinkList.Create;
            (* Initialization of status variables *)
BEGIN
  INHERITED Create;
  FStartNode := NIL;
  FEndNode   := NIL;
  FCount     := 0;
  (* OK, delete it, if you want ... *)
END;


DESTRUCTOR  TLinkList.Destroy;
            (* Delete all remaining nodes and destroy list object *)
BEGIN
  Clear;
  INHERITED Destroy;
END;


FUNCTION  TLinkList.IsEmpty : BOOLEAN;
          (* Returns TRUE when list is empty.
             Same as "CountNodes=0" but faster *)
BEGIN
  IsEmpty := (FStartNode = NIL);
END;


FUNCTION  TLinkList.CountNodes : LONGINT;
          (* Counts the number of nodes in the list by scanning through the list.
             If you manipulate the list *ONLY* with TLinkList methods,
             you can use the "Count" property which is MUCH faster *)
VAR
  CurNode : TListNode;
BEGIN
  FCount  := 0;
  CurNode := FStartNode;
  WHILE CurNode <> NIL DO BEGIN
    INC (FCount);
    CurNode := CurNode.FNext;
    END;
  CountNodes := FCount;
END;


FUNCTION  TLinkList.At (Index0  : LONGINT)   : TListNode;
          (* This function does about the same as the old BPW style TCollection.At:
             It returns the node at the Index0-th position of the list.
             Index0 is zero-based, so for Index0=0, the first node of
             the list is returned.
             Returns NIL if Index0 is less than zero or larger than
             the number of nodes in the list (minus 1) *)
VAR
  CurIndex : LONGINT;
  CurNode  : TListNode;
BEGIN
  At       := NIL;
  IF Index0 < 0 THEN EXIT;
  CurNode  := FStartNode;
  CurIndex := 0;
  WHILE CurNode <> NIL DO BEGIN
    IF CurIndex = Index0 THEN BEGIN
      At := CurNode;
      BREAK;
      END;
    INC (CurIndex);
    CurNode := CurNode.FNext;
    END;
END;


FUNCTION  TLinkList.IndexOf (Node : TListNode) : LONGINT;
          (* Returns the zero-based index of node "Node" in the list.
             Returns -1 when node cannot be found in list *)
VAR
  CurIndex : LONGINT;
  CurNode  : TListNode;
BEGIN
  IndexOf  := -1;
  CurNode  := FStartNode;
  CurIndex := 0;
  WHILE CurNode <> NIL DO BEGIN
    IF CurNode = Node THEN BEGIN
      IndexOf := CurIndex;
      BREAK;
      END;
    INC (CurIndex);
    CurNode := CurNode.FNext;
    END;
END;


PROCEDURE TLinkList.Append (Node : TListNode);
          (* Appends node "Node" to the END of the list *)
BEGIN
  Insert (FEndNode, Node);
END;


PROCEDURE TLinkList.Insert (InsertAfterNode, Node : TListNode);
          (* Inserts node "Node" after node "InsertAfterNode" in the list.
             When InsertAfterNode is NIL, the node will become the
             first of the list. *)
BEGIN
  IF InsertAfterNode = NIL THEN BEGIN   (* Inserts at beginning of list *)
    Node.FNext := FStartNode;
    Node.FPrev := NIL;
    FStartNode := Node;
    END
  ELSE BEGIN                            (* Inserts somewhere in the middle *)
    Node.FNext            := InsertAfterNode.FNext;
    Node.FPrev            := InsertAfterNode;
    InsertAfterNode.FNext := Node;
    END;
  IF Node.FNext = NIL                   (* Append? *)
    THEN FEndNode         := Node
    ELSE Node.FNext.FPrev := Node;
  INC (FCount);
END;


PROCEDURE TLinkList.InsertFirst (Node : TListNode);
          (* Appends node "Node" at the BEGINNING of the list *)
BEGIN
  Insert (NIL, Node);
END;


PROCEDURE TLinkList.InsertList  (InsertAfterNode : TListNode; TheList : TLinkList);
          (* Inserts all nodes of list "TheList" after "InsertAfterNode".
             "TheList" is empty after this call *)
VAR
  S, E        : TListNode;
  CountOfList : LONGINT;
BEGIN
  IF TheList.IsEmpty THEN EXIT;

  S := TheList.FStartNode;
  E := TheList.FEndNode;
  CountOfList := TheList.FCount;
  TheList.FStartNode := NIL;
  TheList.FEndNode   := NIL;
  TheList.FCount     := 0;

  IF IsEmpty THEN BEGIN
    FStartNode := S;
    FEndNode   := E;
    FCount     := CountOfList;
    EXIT;
    END;

  IF InsertAfterNode = NIL THEN BEGIN
    S.FPrev          := NIL;
    E.FNext          := FStartNode;
    FStartNode.FPrev := E;
    FStartNode       := S;
    END
  ELSE BEGIN
    IF InsertAfterNode = FEndNode
      THEN FEndNode := E
      ELSE InsertAfterNode.FNext.FPrev := E;
    E.FNext  := InsertAfterNode.FNext;
    S.FPrev  := InsertAfterNode;
    InsertAfterNode.FNext := S;
    END;
  FCount := FCount + CountOfList;
END;


PROCEDURE TLinkList.Extract (Node : TListNode);
          (* Extracts node "Node" from the list.
             "Node" is NOT destroyed! Its Next and Prev pointers are initialized to NIL *)
BEGIN
  IF Node = FStartNode
    THEN FStartNode       := Node.FNext
    ELSE Node.FPrev.FNext := Node.FNext;
  IF Node = FEndNode
    THEN FEndNode         := Node.FPrev
    ELSE Node.FNext.FPrev := Node.FPrev;
  Node.FNext := NIL;
  Node.FPrev := NIL;
  DEC (FCount);
END;


PROCEDURE TLinkList.Delete (Node : TListNode);
          (* Extracts node "Node" from the list AND destroys it *)
BEGIN
  Extract (Node);
  Node.Free;
END;


PROCEDURE TLinkList.Clear;
          (* Clears the list by calling "Delete" for all nodes.
             So all nodes are destroyed *)
BEGIN
  WHILE NOT IsEmpty DO Delete (FStartNode);
  FStartNode := NIL;   (* This is German overengineering :-) *)
  FEndNode   := NIL;   
  FCount     := 0;     
END;


FUNCTION  TLinkList.Compare (Node1, Node2: TListNode) : INTEGER;
          // virtual Method to compare two nodes (which must not be necessarily
          // contained in the list!)
          // Must be overridden by the application
          // Compares Node1 and Node2, which can only be done by typecasting Node1 and
          // Node2 (which are of type TListNode) to the derived (child) type.
          //    Return value         < 0   if Node1 < Node2
          //                         = 0   if Node1 = Node2
          //                         > 0   if Node1 > Node2
BEGIN
  Result := 0;
END;


PROCEDURE TLinkList.Ensort (Node : TListNode);
          (* Ensorts "Node" to the list. Compare is called to determine the position of
             the new node in the list *)
VAR
  CurNode : TListNode;
BEGIN
  CurNode := FEndNode;
  WHILE CurNode <> NIL DO BEGIN
    IF Compare (CurNode, Node) <= 0 THEN BEGIN
      Insert (CurNode, Node);
      EXIT;
      END;
    CurNode := CurNode.FPrev;
    END;
  Insert (NIL, Node);
END;


          (*
FUNCTION  TLinkList.Compare     (Node1, Node2 : TListNode) : INTEGER;
          Abstract Virtual function for comparisons needed by Ensort.
          You MUST override this function if you want to use Ensort.
          Compares nodes Node1 and Node2 for their order in the list.
          Function result:     < 0    if  Node1 < Node2
                               = 0    if  Node1 = Node2
                               > 0    if  Node1 > Node2  *)


PROCEDURE TLinkList.StartScan (VAR ScanNode);
          (* Starts a scan through the list. ScanNode must be a TLinkList pointer
             or a descendant. It will always point to the next node of the scan *)
BEGIN
  POINTER (ScanNode) := NIL;
END;


FUNCTION  TLinkList.Scan (VAR ScanNode) : BOOLEAN;
          (* Scans to the next node in the list.
             Pass the same variable that you passed to "StartScan".
             Result is FALSE when end of list is reached.
             With these functions you can realize scans in this form:
             VAR
               CurNode : TListNode;   { or a descendant type }
             BEGIN
               ...
               StartScan (CurNode);
               WHILE Scan (CurNode) DO BEGIN
                 { Work with CurNode }
                 END;
               ...
             END;                                       *)
VAR
  TheNode : TListNode ABSOLUTE ScanNode;
BEGIN
  IF (TheNode = FEndNode) OR (FStartNode = NIL) THEN BEGIN
    Scan := FALSE;
    EXIT;
    END;
  Scan := TRUE;
  IF TheNode = NIL
    THEN TheNode := FStartNode
    ELSE TheNode := TheNode.FNext;
END;

END.
